﻿using System.Runtime.InteropServices;
#if !(NETFX_CORE || NETCOREAPP || WINDOWS_UWP)
using Microsoft.Win32.SafeHandles;
using System.Security.Permissions;
using System.Runtime.ConstrainedExecution;
#endif
using System.Text;

internal class CamL : IDisposable
{
    public const int AETARGET_MIN = 16;
    public const int AETARGET_MAX = 220;
    public const int AETARGET_DEF = 120;

    /*
    * Bitdepth: 8~16
    */
    public const int BITDEPTH_MIN = 8;
    public const int BITDEPTH_MAX = 16;

    /*
    * Bin:       1~200
    * bin < 100: bin x bin
    * bin > 100: (bin / 100) x (bin % 100)
    */
    public const int BIN_MIN = 1;
    public const int BIN_MAX = 1000;

    public enum eFEATURE : int
    {
        FEATURE_BINSKIP = 0x01,        /* bin/skip mode */
        FEATURE_HIGH_FULLWELL = 0x02,  /* high fullwell capacity */
        FEATURE_LOW_NOISE = 0x03,      /* low noise mode (Higher signal noise ratio, lower frame rate) */
        FEATURE_GLOBAL_RESET = 0x04,   /* global reset mode */
        FEATURE_CDS_MODE = 0x05        /* CDS Mode: 0 = off, 1 = on, same as CamL_SetCdsMode */
    }

    public enum ePORT : int
    {
        PORT_VR = 0x01,                /* external input, 3V~3.3V*/
        PORT_SUBPV = 0x02,             /* external input, 3.2V~3.5V*/
        PORT_VBLM = 0x03               /* internal generation, supports external regulation, 0V~2.0V*/
    }

    [DllImport("ntdll.dll", ExactSpelling = true, CallingConvention = CallingConvention.Cdecl, SetLastError = false)]
    public static extern void memcpy(IntPtr dest, IntPtr src, IntPtr count);

    public static int TDIBWIDTHBYTES(int bits)
    {
        return ((bits + 31) & (~31)) / 8;
    }

    public void Close()
    {
        Dispose();
    }

    public void Dispose()  // Follow the Dispose pattern - public nonvirtual.
    {
        Dispose(true);
        map_.Remove(id_);
        GC.SuppressFinalize(this);
    }

    /*
    * Open/Close COM
    * protNum:
    *       Windows: COM Port Number, 0 => COM0, 1 => COM1, 2 => COM2, ...
    *       Linux: Camera Link card index, 0, 1, 2, ...
    * baudRate <= 0 means to use default (115200)
    */
    public static CamL Init(int portNum, int baudRate)
    {
        SafeCamLHandle tmphandle = CamL_Init(portNum, baudRate);
        if (tmphandle == null || tmphandle.IsInvalid || tmphandle.IsClosed)
            return null;
        return new CamL(tmphandle);
    }

    /*
    * FPGA Version
    */
    public bool GetFpgaVersion(out string fpgaVer, out string fpgaDate)
    {
        fpgaVer = string.Empty;
        fpgaDate = string.Empty;
        if (!ok())
            return false;
        StringBuilder ver = new StringBuilder(16);
        StringBuilder date = new StringBuilder(16);
        bool result = CamL_GetFpgaVersion(handle_, ver, date);
        if (result)
        {
            fpgaVer = ver.ToString();
            fpgaDate = date.ToString();
        }
        return result;
    }

    /*
    * Model
    */
    public bool GetModel(out string model)
    {
        if (!ok())
        {
            model = string.Empty;
            return false;
        }
        StringBuilder mn = new StringBuilder(32);
        bool result = CamL_GetModel(handle_, mn);
        model = mn.ToString();
        return result;
    }

    /*
    * Product Number
    */
    public void SetPN(string pn)
    {
        if (!ok())
            return;
        if (string.IsNullOrEmpty(pn))
            throw new ArgumentNullException(nameof(pn));

        CamL_SetPN(handle_, pn);
    }

    public bool GetPN(out string pn)
    {
        if (!ok())
        {
            pn = string.Empty;
            return false;
        }
        StringBuilder str = new StringBuilder(64);
        bool result = CamL_GetPN(handle_, str);
        pn = str.ToString();
        return result;
    }

    /*
    * Series Number
    */
    public void SetSN(string sn)
    {
        if (!ok())
            return;
        if (string.IsNullOrEmpty(sn))
            throw new ArgumentNullException(nameof(sn));

        CamL_SetSN(handle_, sn);
    }

    public bool GetSN(out string sn)
    {
        if (!ok())
        {
            sn = string.Empty;
            return false;
        }
        StringBuilder str = new StringBuilder(64);
        bool result = CamL_GetSN(handle_, str);
        sn = str.ToString();
        return result;
    }

    /*
    * Auto Exposure
    */
    public bool AutoExposure
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsAutoExposureEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableAutoExposure(handle_, value);
        }
    }

    /*
    * Conversion Gain
    */
    public int CG
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCG(handle_);
        }
        set
        {
            if (ok())
                CamL_SetCG(handle_, value);
        }
    }

    public void GetCGRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetCGRange(handle_, out min, out  max, out def);
    }

    /*
    * Exposure Read Mode
    * mode: 0 = ITR, 1 = IWR
    */
    public bool IsSupportExpoReadMode
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportExpoReadMode(handle_);
        }
    }

    public int ExpoReadMode
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetExpoReadMode(handle_);
        }
        set
        {
            if (ok())
                CamL_SetExpoReadMode(handle_, value);
        }
    }

    /*
    * Exposure Time
    * Unit: us
    */
    public uint ExpoTime
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetExpoTime(handle_);
        }
        set
        {
            if (ok())
                CamL_SetExpoTime(handle_, value);
        }
    }

    public void GetExpoTimeRange(out uint min, out uint max, out uint def)
    {
        if (!ok())
        {  
            min = max = def = 0;
            return;
        }
        CamL_GetExpoTimeRange(handle_, out min, out max, out def);
    }

    /*
    * Sequence Exposure Time
    * Unit: us
    */
    public bool IsSupportSeqExpoTime
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportSeqExpoTime(handle_);
        }
    }


    public bool SeqExpoTimeEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSeqExpoTimeEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableSeqExpoTime(handle_, value);
        }
    }

    public uint SeqExpoTimeCount
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetSeqExpoTimeCount(handle_);
        }
        set
        {
            if (ok())
                CamL_SetSeqExpoTimeCount(handle_, value);
        }
    }

    public void SetSeqExpoTime(int index, uint time)
    {
        if (!ok())
            return;
        if (index < 0 || index >= SeqExpoTimeCount)
            throw new ArgumentOutOfRangeException(nameof(index));
        CamL_SetSeqExpoTime(handle_, index, time);
    }

    public uint GetSeqExpoTime(int index)
    {
        if(!ok())
            return 0;
        if (index < 0 || index >= SeqExpoTimeCount)
            throw new ArgumentOutOfRangeException(nameof(index));
        return CamL_GetSeqExpoTime(handle_, index);
    }

    /*
    * Exposure Gain
    * Unit: %
    */
    public uint ExpoGain
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetExpoGain(handle_);
        }
        set
        {
            if (ok())
                CamL_SetExpoGain(handle_, value);
        }
    }

    public void GetExpoGainRange(out uint min, out uint max, out uint def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetExpoGainRange(handle_, out min, out max, out def);
    }

    /*
    * Frame Rate
    */
    public int FrameRate
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetFrameRate(handle_);
        }
        set
        {
            if (ok())
                CamL_SetFrameRate(handle_, value);
        }
    }

    public int MaxFrameRate
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetMaxFrameRate(handle_);
        }
    }

    /*
    * Image Size
    */
    public void GetSize(out int width, out int height)
    {
        if (!ok())
        {
            width = height = 0;
            return;
        }
        CamL_GetSize(handle_, out width, out height);
    }

    /*
    * ROI
    */
    public void SetRoi(int xOffset, int yOffset, int width, int height)
    {
        if (!ok())
            return;
        CamL_SetRoi(handle_, xOffset, yOffset, width, height);
    }

    public void GetRoi(out int xOffset, out int yOffset, out int width, out int height)
    {
        if (!ok())
        {
            xOffset = yOffset = width = height = 0;
            return;
        }
        CamL_GetRoi(handle_, out xOffset, out yOffset, out width, out height);
    }

    public bool IsSupportRoiN
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportRoiN(handle_);
        }
    }

    public void SetRoiN(uint[] xOffset, uint[] yOffset, uint[] xWidth, uint[] yHeight, uint hNum, uint vNum)
    {
        if (!ok())
            return;
        CamL_SetRoiN(handle_, xOffset, yOffset, xWidth, yHeight, hNum, vNum);
    }

    public void GetRoiN(uint[] xOffset, uint[] yOffset, uint[] xWidth, uint[] yHeight, out uint hNum, out uint vNum)
    {
        if (!ok())
        {
            hNum = vNum = 0;
            return;
        }
        CamL_GetRoiN(handle_, xOffset, yOffset, xWidth, yHeight, out hNum, out vNum);
    }

    /*
    * FPN Correction
    */
    public bool FpncEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsFpncEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableFpnc(handle_, value);
        }
    }

    public int FpncThreshold
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetFpncThreshold(handle_);
        }
        set
        {
            if (ok())
                CamL_SetFpncThreshold(handle_, value);
        }
    }

    public void GetFpncThresholdRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetFpncThresholdRange(handle_, out min, out max, out def);
    }

    public bool WriteFpncData(int index, byte[] data, int length, DelegateProgressCallback callback)
    {
        if (!ok())
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_WriteFpncData(handle_, index, data, length, cb_, id_);
        }
        else
        {
            return CamL_WriteFpncData(handle_, index, data, length, null, id_);
        }
    }

    public bool ReadFpncData(int index, byte[] data, int length, DelegateProgressCallback callback)
    {
        if (!ok())
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_ReadFpncData(handle_, index, data, length, cb_, id_);
        }
        else
        {
            return CamL_ReadFpncData(handle_, index, data, length, null, id_);
        }
    }

    public bool AutoFpncEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsAutoFpncEnable(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableAutoFpnc(handle_, value);
        }
    }

    public int ManualFpncIndex
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetManualFpncIndex(handle_);
        }
        set
        {
            if (ok())
                CamL_SetManualFpncIndex(handle_, value);
        }
    }

    public void GetFpncIndexRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetFpncIndexRange(handle_, out min, out max, out def);
    }

    /*
    * Flat Field Correction
    */
    public bool FfcEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsFfcEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableFfc(handle_, value);
        }
    }

    public bool AutoFfcEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsAutoFfcEnable(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableAutoFfc(handle_, value);
        }
    }

    public int ManualFfcIndex
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetManualFfcIndex(handle_);
        }
        set
        {
            if (ok())
                CamL_SetManualFfcIndex(handle_, value);
        }
    }

    public void GetFfcIndexRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetFfcIndexRange(handle_, out min, out max, out def);
    }

    public bool WriteFfcData(int index, byte[] data, int length, DelegateProgressCallback callback)
    {
        if (!ok())
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_WriteFfcData(handle_, index, data, length, cb_, id_);
        }
        else
        {
            return CamL_WriteFfcData(handle_, index, data, length, null, id_);
        }
    }

    /*
    * Defect Pixel Correction
    */
    public bool DfcEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsDfcEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableDfc(handle_, value);
        }
    }

    public int DfcThreshold
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetDfcThreshold(handle_);
        }
        set
        {
            if (ok())
                CamL_SetDfcThreshold(handle_, value);
        }
    }

    public void GetDfcThresholdRange(out int min, out int max, out int def)
    {
        CamL_GetDfcThresholdRange(out min, out max, out def);
    }

    public bool WriteDfcData(int index, byte[] data, int length, DelegateProgressCallback callback)
    {
        if (!ok())
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_WriteDfcData(handle_, index, data, length, cb_, id_);
        }
        else
        {
            return CamL_WriteDfcData(handle_, index, data, length, null, id_);
        }
    }

    public bool ReadDfcData(int index, byte[] data, int length, DelegateProgressCallback callback)
    {
        if (!ok())
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_ReadDfcData(handle_, index, data, length, cb_, id_);
        }
        else
        {
            return CamL_ReadDfcData(handle_, index, data, length, null, id_);
        }
    }

    public bool AutoDfcEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsAutoDfcEnable(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableAutoDfc(handle_, value);
        }
    }

    public int ManualDfcIndex
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetManualDfcIndex(handle_);
        }
        set
        {
            if (ok())
                CamL_SetManualDfcIndex(handle_, value);
        }
    }

    public void GetDfcIndexRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetDfcIndexRange(handle_, out min, out max, out def);
    }

    /*
    * Defect Pixel Correction
    */
    public bool DpcEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsDpcEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableDpc(handle_, value);
        }
    }

    /*
    * Denoise
    */
    public bool DenoiseEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsDenoiseEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableDenoise(handle_, value);
        }
    }

    public int Denoise
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetDenoise(handle_);
        }
        set
        {
            if (ok())
                CamL_SetDenoise(handle_, value);
        }
    }

    public void GetDenoiseRange(out int min, out int max, out int def)
    {
        CamL_GetDenoiseRange(out min, out max, out def);
    }

    /*
    * Sharpen
    */
    public int Sharpen
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetSharpen(handle_);
        }
        set
        {
            if (ok())
                CamL_SetSharpen(handle_, value);
        }
    }

    public void GetSharpenRange(out int min, out int max, out int def)
    {
        CamL_GetSharpenRange(out min, out max, out def);
    }

    /*
    * Temperature Control
    */
    public float TargetTemperature
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetTargetTemperature(handle_);
        }
        set
        {
            if (ok())
                CamL_SetTargetTemperature(handle_, value);
        }
    }

    public float CurrentTemperature
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCurrentTemperature(handle_);
        }
    }

    public bool TecEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsTecEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableTec(handle_, value);
        }
    }

    public bool FanEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsFanEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableFan(handle_, value);
        }
    }

    public int Fan
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetFan(handle_);
        }
        set
        {
            if (ok())
                CamL_SetFan(handle_, value);
        }
    }

    public void GetFanRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetFanRange(handle_, out min, out max, out def);
    }

    /*
    * Trigger
    */
    public bool TriggerEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsTriggerEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_EnableTrigger(handle_, value);
        }
    }

    public int TriggerSource
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetTriggerSource(handle_);
        }
        set
        {
            if (ok())
                CamL_SetTriggerSource(handle_, value);
        }
    }

    public bool SoftAlwaysEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSoftAlwaysEnabled(handle_);
        }
        set
        {
            if (ok())
                CamL_SetSoftAlwaysEnabled(handle_, value);
        }
    }

    public int TriggerProhibitedTime
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetTriggerProhibitedTime(handle_);
        }
        set
        {
            if (ok())
                CamL_SetTriggerProhibitedTime(handle_, value);
        }
    }

    public int InputActivation
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetInputActivation(handle_);
        }
        set
        {
            if (ok())
                CamL_SetInputActivation(handle_, value);
        }
    }

    public int BurstCounter
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetBurstCounter(handle_);
        }
        set
        {
            if (ok())
                CamL_SetBurstCounter(handle_, value);
        }
    }

    public int CounterSource
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCounterSource(handle_);
        }
        set
        {
            if (ok())
                CamL_SetCounterSource(handle_, value);
        }
    }

    public int CounterValue
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCounterValue(handle_);
        }
        set
        {
            if (ok())
                CamL_SetCounterValue(handle_, value);
        }
    }

    public int PwmSource
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetPwmSource(handle_);
        }
        set
        {
            if (ok())
                CamL_SetPwmSource(handle_, value);
        }
    }

    public int IoDir
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetIoDir(handle_);
        }
        set
        {
            if (ok())
                CamL_SetIoDir(handle_, value);
        }
    }

    public void SetTriggerDelay(int ioLineNumber, int value)
    {
        if (!ok())
            return;
        CamL_SetTriggerDelay(handle_, ioLineNumber, value);
    }

    public int GetTriggerDelay(int ioLineNumber)
    {
        if (!ok())
            return 0;
        return CamL_GetTriggerDelay(handle_, ioLineNumber);
    }

    public void SetOutputMode(int ioLineNumber, int value)
    {
        if (!ok())
            return;
        CamL_SetOutputMode(handle_, ioLineNumber, value);
    }

    public int GetOutputMode(int ioLineNumber)
    {
        if (!ok())
            return 0;
        return CamL_GetOutputMode(handle_, ioLineNumber);
    }

    public int StrobDuration
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetStrobDuration(handle_);
        }
        set
        {
            if (ok())
                CamL_SetStrobDuration(handle_, value);
        }
    }

    public int StrobPreDelay
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetStrobPreDelay(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetStrobPreDelay(handle_, value);
        }
    }

    public int StrobDelay
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetStrobDelay(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetStrobDelay(handle_, value);
        }
    }

    public int UserValue
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetUserValue(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetUserValue(handle_, value);
        }
    }

    public void ResetCounter()
    {
        if (!ok())
            return;
        CamL_ResetCounter(handle_);
    }

    public void SetDebouncerTime(int ioLineNumber, int value)
    {
        if (!ok())
            return;
        CamL_SetDebouncerTime(handle_, ioLineNumber, value);
    }

    public int GetDebouncerTime(int ioLineNumber)
    {
        if (!ok())
            return 0;
        return CamL_GetDebouncerTime(handle_, ioLineNumber);
    }

    public void EnableOutputInverter(int ioLineNumber, bool bInverter)
    {
        if (!ok())
            return;
        CamL_EnableOutputInverter(handle_, ioLineNumber, bInverter);
    }

    public bool IsOutputInverter(int ioLineNumber)
    {
        if (!ok())
            return false;
        return CamL_IsOutputInverter(handle_, ioLineNumber);
    }

    public int OutputCounterValue
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetOutputCounterValue(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetOutputCounterValue(handle_, value);
        }
    }

    public int OutputPause
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetOutputPause(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetOutputPause(handle_, value);
        }
    }

    public void Trigger(bool bStart)
    {
        if (!ok())
            return;
        CamL_Trigger(handle_, bStart);
    }

    public void ClearFrameCounter()
    {
        if (!ok())
            return;
        CamL_ClearFrameCounter(handle_);
    }

    public void ClearTriggerCounter()
    {
        if (!ok())
            return;
        CamL_ClearTriggerCounter(handle_);
    }

    public void ClearTimeStampCounter()
    {
        if (!ok())
            return;
        CamL_ClearTimeStampCounter(handle_);
    }

    public void ClearAllCounter()
    {
        if (!ok())
            return;
        CamL_ClearAllCounter(handle_);
    }

    public bool CounteEnabled
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsCounteEnabled(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_EnableCounte(handle_, value);
        }
    }

    public void SetNucCtrl(int fpnc_en, int bias_en, int denoise_en)
    {
        if (!ok())
            return;
        CamL_SetNucCtrl(handle_, fpnc_en, bias_en, denoise_en);
    }

    public int GetNucCtrl()
    {
        if (!ok())
            return 0;
        return CamL_GetNucCtrl(handle_);
    }

    public bool VFlip
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsVFlip(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetVFlip(handle_, value);
        }
    }

    public bool HFlip
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsHFlip(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetHFlip(handle_, value);
        }
    }

    public bool IsSupportWorkMode
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportWorkMode(handle_);
        }
    }

    public int WorkMode
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetWorkMode(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetWorkMode(handle_, value);
        }
    }

    public bool SetAutoExposure(uint minTime, uint maxTime, uint dampTime, uint minGain, uint maxGain, uint dampGain, uint policy, uint peakPercent, uint overPolicy)
    {
        if (!ok())
            return false;
        return CamL_SetAutoExposure(handle_, minTime, maxTime, dampTime, minGain, maxGain, dampGain, policy, peakPercent, overPolicy);
    }

    public bool GetAutoExposure(out uint minTime, out uint maxTime, out uint dampTime, out uint minGain, out uint maxGain, out uint dampGain, out uint policy, out uint peakPercent, out uint overPolicy)
    {
        if (!ok())
        {
            minTime = maxTime = dampTime = minGain = maxGain = dampGain = policy = peakPercent = overPolicy = 0;
            return false;
        }
        return CamL_GetAutoExposure(handle_, out minTime, out maxTime, out dampTime, out minGain, out maxGain, out dampGain, out policy, out peakPercent, out overPolicy);
    }

    /*
    * Bitdepth
    */
    public int Bitdepth
    {
        get
        {
            if (!ok())
                return -1;
            return CamL_GetBitdepth(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetBitdepth(handle_, value);
        }
    }

    public bool IsSupportBitdepth(int bitdepth)
    {
        if (!ok())
            return false;
        return CamL_IsSupportBitdepth(handle_, bitdepth);
    }

    /*
    * Bin
    */
    public int Bin
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetBin(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetBin(handle_, value);
        }
    }

    public bool IsSupportBin(int bin)
    {
        if (!ok())
            return false;
        return CamL_IsSupportBin(handle_, bin);
    }

    public void GetBinSize(int bin, out int width, out int height)
    {
        if (!ok())
        {
            width = height = 0;
            return;
        }
        CamL_GetBinSize(handle_, bin, out width, out height);
    }

    /*
    * Bin Mode
    */
    public int BinMode
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetBinMode(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetBinMode(handle_, value);
        }
    }

    /*
    * Output Image Mode
    */
    public bool IsSupportOutputImage
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportOutputImage(handle_);
        }
    }

    public int OutputImage
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetOutputImage(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetOutputImage(handle_, value);
        }
    }

    /*
    * Feature
    */
    public bool SetFeature(eFEATURE feature, int val)
    {
        if (!ok())
            return false;
        return CamL_SetFeature(handle_, feature, val);
    }

    public int GetFeature(eFEATURE feature)
    {
        if (!ok())
            return 0;
        return CamL_GetFeature(handle_, feature);
    }

    public bool IsSupportFeature(eFEATURE feature)
    {
        if (!ok())
            return false;
        return CamL_IsSupportFeature(handle_, feature);
    }

    /*
    * TReady
    */
    public int TReady
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetTReady(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetTReady(handle_, value);
        }
    }

    /*
    * Tcds
    */
    public int Tcds
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetTcds(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetTcds(handle_, value);
        }
    }

    public void GetTcdsRange(out int min, out int max, out int def)
    {
        if (!ok())
        {
            min = max = def = 0;
            return;
        }
        CamL_GetTcdsRange(handle_, out min, out max, out def);
    }

    /*
    * CDS Mode
    */
    public int CdsMode
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCdsMode(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetCdsMode(handle_, value);
        }
    }

    /*
    * Save/Load Parameters
    */
    public bool SaveParameterToCamera()
    {
        if (!ok())
            return false;
        return CamL_SaveParameterToCamera(handle_);
    }

    public void LoadParameterFromCamera()
    {
        if (!ok())
            return;
        CamL_LoadParameterFromCamera(handle_);
    }

    public void FactorySettingParameter()
    {
        if (!ok())
            return;
        CamL_FactorySettingParameter(handle_);
    }

    /*
    * Firmware Update
    */
    public bool Update(string ufwFilePath, DelegateProgressCallback callback)
    {
        if (handle_ == null || handle_.IsInvalid || handle_.IsClosed)
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_Update(handle_, ufwFilePath, cb_, id_);
        }
        else
        {
            return CamL_Update(handle_, ufwFilePath, null, id_);
        }
    }

    /*
    * DDR
    */
    public bool IsSupportDDR
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportDDR(handle_);
        }
    }

    public bool WriteDDR(uint addr, IntPtr data, int length, DelegateProgressCallback callback)
    {
        if (handle_ == null || handle_.IsInvalid || handle_.IsClosed)
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_WriteDDR(handle_, addr, data, length, cb_, id_);
        }
        else
        {
            return CamL_WriteDDR(handle_, addr, data, length, null, id_);
        }
    }

    /*
    * eMMC
    */
    public bool IsSupportEMMC
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportEMMC(handle_);
        }
    }

    public bool WriteEMMC(uint addr, IntPtr data, int length, DelegateProgressCallback callback)
    {
        if (handle_ == null || handle_.IsInvalid || handle_.IsClosed)
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_WriteEMMC(handle_, addr, data, length, cb_, id_);
        }
        else
        {
            return CamL_WriteEMMC(handle_, addr, data, length, null, id_);
        }
    }

    public bool ReadEMMC(uint addr, IntPtr data, int length, DelegateProgressCallback callback)
    {
        if (handle_ == null || handle_.IsInvalid || handle_.IsClosed)
            return false;
        funP_ = callback;
        if (callback != null)
        {
            cb_ = new PROGRESS_CALLBACK(ProgressCallback);
            return CamL_ReadEMMC(handle_, addr, data, length, cb_, id_);
        }
        else
        {
            return CamL_ReadEMMC(handle_, addr, data, length, null, id_);
        }
    }

    /*
    * Lane
    */
    public bool IsSupportLane
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportLane(handle_);
        }
    }

    public int Lane
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetLane(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetLane(handle_, value);
        }
    }

    /*
    * Cable
    */
    public bool IsSupportCable
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportCable(handle_);
        }
    }

    public int CableNumber
    {
        get
        {
            if (!ok())
                return 0;
            return CamL_GetCableNumber(handle_);
        }
        set
        {
            if (!ok())
                return;
            CamL_SetCableNumber(handle_, value);
        }
    }

    /*
    * Voltage
    */
    public bool SetVoltage(ePORT port, int cg, int val)
    {
        if (!ok())
            return false;
        return CamL_SetVoltage(handle_, port, cg, val);
    }

    public int GetVoltage(ePORT port, int cg)
    {
        if (!ok())
            return 0;
        return CamL_GetVoltage(handle_, port, cg);
    }

    public bool IsSupportVoltage(ePORT port)
    {
        if (!ok())
            return false;
        return CamL_IsSupportVoltage(handle_, port);
    }

    /*
    * Shutter
    */
    public int Shutter
    {
        set
        {
            if (!ok())
                return;
            CamL_SetShutter(handle_, value);
        }
        get
        {
            if (!ok())
                return 0;
            return CamL_GetShutter(handle_);
        }
    }

    public bool IsSupportShutter
    {
        get
        {
            if (!ok())
                return false;
            return CamL_IsSupportShutter(handle_);
        }
    }

    private static int sid_ = 0;
    private static Dictionary<int, Object> map_ = new Dictionary<int, Object>();

    private SafeCamLHandle handle_;
    private int id_;
    private DelegateProgressCallback funP_;
    private PROGRESS_CALLBACK cb_;

    private bool ok()
    {
        if (handle_ == null || handle_.IsInvalid || handle_.IsClosed)
            return false;
        return true;
    }

    private CamL(SafeCamLHandle h)
    {
        handle_ = h;
        id_ = Interlocked.Increment(ref sid_);
        map_.Add(id_, this);
    }

    ~CamL()
    {
        Dispose(false);
    }

#if NETCOREAPP
    private const string dll = "CameraLinkCmd";
#elif LINUX
    private const string dll = "libCameraLinkCmd.so";
#else
    private const string dll = "CameraLinkCmd.dll";
#endif

    protected virtual void Dispose(bool disposing)
    {
        // Note there are three interesting states here:
        // 1) CreateFile failed, _handle contains an invalid handle
        // 2) We called Dispose already, _handle is closed.
        // 3) _handle is null, due to an async exception before
        //    calling CreateFile. Note that the finalizer runs
        //    if the constructor fails.
        if (handle_ != null && !handle_.IsInvalid)
        {
            // Free the handle
            handle_.Dispose();
        }
        // SafeHandle records the fact that we've called Dispose.
    }

    private static void ProgressCallback(IntPtr ctx, int result, int step)
    {
        Object obj = null;
        if (map_.TryGetValue(ctx.ToInt32(), out obj) && (obj != null))
        {
            CamL pthis = obj as CamL;
            if ((pthis != null) && (pthis.funP_ != null))
                pthis.funP_(result, step);
        }
    }

#if !(NETFX_CORE || NETCOREAPP || WINDOWS_UWP)
    public class SafeCamLHandle : SafeHandleZeroOrMinusOneIsInvalid
    {
        [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Winapi)]
        private static extern void CamL_UnInit(IntPtr h);

        public SafeCamLHandle()
            : base(true)
        {
        }

        [ReliabilityContract(Consistency.WillNotCorruptState, Cer.MayFail)]
        override protected bool ReleaseHandle()
        {
            // Here, we must obey all rules for constrained execution regions.
            CamL_UnInit(handle);
            return true;
        }
    };
#else
    public class SafeCamLHandle : SafeHandle
    {
        [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Winapi)]
        private static extern void CamL_UnInit(IntPtr h);
        
        public SafeCamLHandle()
            : base(IntPtr.Zero, true)
        {
        }
        
        override protected bool ReleaseHandle()
        {
            CamL_UnInit(handle);
            return true;
        }
        
        public override bool IsInvalid
        {
            get { return base.handle == IntPtr.Zero || base.handle == (IntPtr)(-1); }
        }
    };
#endif
    public delegate void DelegateProgressCallback(int result, int step);
    [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
    public delegate void PROGRESS_CALLBACK(IntPtr ctx, int result, int step);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern SafeCamLHandle CamL_Init(int portNum, int baudRate);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_GetFpgaVersion(SafeCamLHandle h, StringBuilder fpgaVer, StringBuilder fpgaDate);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_GetModel(SafeCamLHandle h, StringBuilder mn);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetPN(SafeCamLHandle h, [MarshalAs(UnmanagedType.LPStr)] string pn);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_GetPN(SafeCamLHandle h, StringBuilder pn);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetSN(SafeCamLHandle h, [MarshalAs(UnmanagedType.LPStr)] string sn);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_GetSN(SafeCamLHandle h, StringBuilder sn);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableAutoExposure(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsAutoExposureEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetCG(SafeCamLHandle h, int cg);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetCG(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetCGRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsSupportExpoReadMode(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetExpoReadMode(SafeCamLHandle h, int mode);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetExpoReadMode(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetExpoTime(SafeCamLHandle h, uint time);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern uint CamL_GetExpoTime(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetExpoTimeRange(SafeCamLHandle h, out uint pMin, out uint pMax, out uint pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsSupportSeqExpoTime(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableSeqExpoTime(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnabled);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsSeqExpoTimeEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetSeqExpoTimeCount(SafeCamLHandle h, uint count);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern uint CamL_GetSeqExpoTimeCount(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetSeqExpoTime(SafeCamLHandle h, int idx, uint time);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern uint CamL_GetSeqExpoTime(SafeCamLHandle h, int idx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetExpoGain(SafeCamLHandle h, uint gain);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern uint CamL_GetExpoGain(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetExpoGainRange(SafeCamLHandle h, out uint pMin, out uint pMax, out uint pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetFrameRate(SafeCamLHandle h, int frameRate);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetFrameRate(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetMaxFrameRate(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetSize(SafeCamLHandle h, out int pWidth, out int pHeight);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetRoi(SafeCamLHandle h, int xOffset, int yOffset, int width, int height);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetRoi(SafeCamLHandle h, out int pxOffset, out int pyOffset, out int pWidth, out int pHeight);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern bool CamL_IsSupportRoiN(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetRoiN(SafeCamLHandle h, [In] uint[] xOffset, [In] uint[] yOffset, [In] uint[] xWidth, [In] uint[] yHeight, uint hNum, uint vNum);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetRoiN(SafeCamLHandle h, [Out] uint[] xOffset, [Out] uint[] yOffset, [Out] uint[] xWidth, [Out] uint[] yHeight, out uint hNum, out uint vNum);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableFpnc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsFpncEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetFpncThreshold(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetFpncThreshold(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetFpncThresholdRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_WriteFpncData(SafeCamLHandle h, int index, byte[] pData, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_ReadFpncData(SafeCamLHandle h, int index, byte[] pData, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableAutoFpnc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsAutoFpncEnable(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetManualFpncIndex(SafeCamLHandle h, int index);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetManualFpncIndex(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetFpncIndexRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableFfc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsFfcEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableAutoFfc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsAutoFfcEnable(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetManualFfcIndex(SafeCamLHandle h, int index);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetManualFfcIndex(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetFfcIndexRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_WriteFfcData(SafeCamLHandle h, int index, byte[] pData, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableDfc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsDfcEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetDfcThreshold(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetDfcThreshold(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetDfcThresholdRange( out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_WriteDfcData(SafeCamLHandle h, int index, byte[] pData, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_ReadDfcData(SafeCamLHandle h, int index, byte[] pData, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableAutoDfc(SafeCamLHandle h, bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsAutoDfcEnable(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetManualDfcIndex(SafeCamLHandle h, int index);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetManualDfcIndex(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetDfcIndexRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableDpc(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsDpcEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableDenoise(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsDenoiseEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetDenoise(SafeCamLHandle h, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetDenoise(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetDenoiseRange(out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetSharpen(SafeCamLHandle h, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetSharpen(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetSharpenRange(out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetTargetTemperature(SafeCamLHandle h, float temp);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern float CamL_GetTargetTemperature(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern float CamL_GetCurrentTemperature(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableTec(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsTecEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableFan(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsFanEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetFan(SafeCamLHandle h, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetFan(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_GetFanRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsTriggerEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableTrigger(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetTriggerSource(SafeCamLHandle h, int source);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetTriggerSource(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetSoftAlwaysEnabled(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnabled);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsSoftAlwaysEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetTriggerProhibitedTime(SafeCamLHandle h, int time);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetTriggerProhibitedTime(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetInputActivation(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetInputActivation(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetBurstCounter(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetBurstCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetCounterSource(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetCounterSource(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetCounterValue(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetCounterValue(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetPwmSource(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetPwmSource(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetIoDir(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetIoDir(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetTriggerDelay(SafeCamLHandle h, int ioLineNumber, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetTriggerDelay(SafeCamLHandle h, int ioLineNumber);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetOutputMode(SafeCamLHandle h, int ioLineNumber, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetOutputMode(SafeCamLHandle h, int ioLineNumber);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetStrobDuration(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetStrobDuration(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetStrobPreDelay(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetStrobPreDelay(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetStrobDelay(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetStrobDelay(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetUserValue(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetUserValue(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_ResetCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetDebouncerTime(SafeCamLHandle h, int ioLineNumber, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetDebouncerTime(SafeCamLHandle h, int ioLineNumber);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableOutputInverter(SafeCamLHandle h, int ioLineNumber, [MarshalAs(UnmanagedType.I1)] bool bInverter);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsOutputInverter(SafeCamLHandle h, int ioLineNumber);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetOutputCounterValue(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetOutputCounterValue(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetOutputPause(SafeCamLHandle h, int value);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetOutputPause(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_Trigger(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bStart);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_ClearFrameCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_ClearTriggerCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_ClearTimeStampCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_ClearAllCounter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_EnableCounte(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsCounteEnabled(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetNucCtrl(SafeCamLHandle h, int fpnc_en, int bias_en, int denoise_en);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetNucCtrl(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetVFlip(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bVFlip);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsVFlip(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetHFlip(SafeCamLHandle h, [MarshalAs(UnmanagedType.I1)] bool bHFlip);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsHFlip(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_IsSupportWorkMode(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern void CamL_SetWorkMode(SafeCamLHandle h, int mode);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    private static extern int CamL_GetWorkMode(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_SetAutoExposure(
        SafeCamLHandle h,
        uint minTime,
        uint maxTime,
        uint dampTime,
        uint minGain,
        uint maxGain,
        uint dampGain,
        uint policy,
        uint peakPercent,
        uint overPolicy);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    private static extern bool CamL_GetAutoExposure(
        SafeCamLHandle h,
        out uint minTime,
        out uint maxTime,
        out uint dampTime,
        out uint minGain,
        out uint maxGain,
        out uint dampGain,
        out uint policy,
        out uint peakPercent,
        out uint overPolicy);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetBitdepth(SafeCamLHandle h, int bitdepth);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetBitdepth(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportBitdepth(SafeCamLHandle h, int bitdepth);

    /* Bin */
    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetBin(SafeCamLHandle h, int bin);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetBin(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportBin(SafeCamLHandle h, int bit);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_GetBinSize(SafeCamLHandle h, int bin, out int pWidth, out int pHeight);

    /* Bin Mode */
    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetBinMode(SafeCamLHandle h, int mode);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetBinMode(SafeCamLHandle h);

    /* Output Image Mode */
    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportOutputImage(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetOutputImage(SafeCamLHandle h, int mode);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetOutputImage(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetFeature(SafeCamLHandle h, eFEATURE feature, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetFeature(SafeCamLHandle h, eFEATURE feature);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportFeature(SafeCamLHandle h, eFEATURE feature);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_SetTReady(SafeCamLHandle h, int t_ready);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetTReady(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_SetTcds(SafeCamLHandle h, int tcds);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetTcds(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_GetTcdsRange(SafeCamLHandle h, out int pMin, out int pMax, out int pDef);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_SetCdsMode(SafeCamLHandle h, int cds_mode);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetCdsMode(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SaveParameterToCamera(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_LoadParameterFromCamera(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_FactorySettingParameter(SafeCamLHandle h);

    [DllImport(dll, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
#if WINDOWS
    public static extern bool CamL_Update(SafeCamLHandle h, [MarshalAs(UnmanagedType.LPWStr)] string ufwFilePath, PROGRESS_CALLBACK fun, IntPtr ctx);
#else
    public static extern bool CamL_Update(SafeCamLHandle h, [MarshalAs(UnmanagedType.LPStr)] string ufwFilePath, PROGRESS_CALLBACK fun, IntPtr ctx);
#endif

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportDDR(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_WriteDDR(SafeCamLHandle h, uint addr, IntPtr data, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportEMMC(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_WriteEMMC(SafeCamLHandle h, uint addr, IntPtr data, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_ReadEMMC(SafeCamLHandle h, uint addr, IntPtr data, int length, PROGRESS_CALLBACK fun, IntPtr ctx);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportLane(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_SetLane(SafeCamLHandle h, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetLane(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportCable(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern void CamL_SetCableNumber(SafeCamLHandle h, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetCableNumber(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetVoltage(SafeCamLHandle h, ePORT port, int cg, int val);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetVoltage(SafeCamLHandle h, ePORT port, int cg);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportVoltage(SafeCamLHandle h, ePORT port);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_SetShutter(SafeCamLHandle h, int bEnable);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    public static extern int CamL_GetShutter(SafeCamLHandle h);

    [DllImport(dll, ExactSpelling = true, CallingConvention = CallingConvention.Cdecl)]
    [return: MarshalAs(UnmanagedType.I1)]
    public static extern bool CamL_IsSupportShutter(SafeCamLHandle h);
}
